<?php

namespace App\Http\Controllers;

use App\Models\Report;
use App\Models\Payment;
use App\Models\Transaction;
use App\Models\Refund;
use App\Models\Analytics;
use App\Models\AuditLog;
use App\Models\UserCourse;
use App\Models\JobApplication;
use App\Models\Certification;
use App\Models\CompanyFeatureUsage;
use App\Models\Interview;
use App\Models\CandidateProfile;
use App\Models\UserSkill;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ReportsExport;
use Illuminate\Support\Facades\Log; // Already included
use Carbon\Carbon;

class ReportsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
        $this->middleware('role:Super Admin,Company')->only(['exportReport', 'getHiringAnalytics', 'getCandidateQuality', 'generateReport']);
        $this->middleware('role:Super Admin')->except(['getHiringAnalytics', 'getCandidateQuality', 'getUserEngagement', 'exportReport', 'generateReport']);
    }

    public function index(): JsonResponse
    {
        $reports = Report::with('user')->get();
        return response()->json(['status' => 'success', 'data' => $reports], 200);
    }

    public function show($id): JsonResponse
    {
        $report = Report::with('user')->findOrFail($id);
        return response()->json(['status' => 'success', 'data' => $report], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'nullable|exists:Users,User_id',
            'Report_type' => 'required|in:payment_summary,user_activity',
            'Payment_status' => 'nullable|in:pending,completed,failed',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $data = $request->only(['User_id', 'Report_type', 'Payment_status']);
        $payments = Payment::query();

        if ($data['User_id']) $payments->where('User_id', $data['User_id']);
        if (isset($data['Payment_status'])) $payments->where('Payment_status', $data['Payment_status']);

        $totalAmount = $payments->sum('Amount');
        $transactionCount = $payments->count();

        $report = Report::create([
            'User_id' => $data['User_id'],
            'Report_type' => $data['Report_type'],
            'Total_amount' => $totalAmount,
            'Transaction_count' => $transactionCount,
            'Payment_status' => $data['Payment_status'],
        ]);

        return response()->json(['status' => 'success', 'data' => $report], 201);
    }

    public function generateReport(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'nullable|exists:users,User_id',
            'Payment_status' => 'nullable|in:pending,completed,failed',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed for generateReport: ' . json_encode($validator->errors()));
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $query = Payment::query()
            ->select('Payments.User_id', DB::raw('SUM(Payments.Amount) as total_payments'), DB::raw('COUNT(*) as payment_count'))
            ->leftJoin('Transactions', 'Payments.Payment_id', '=', 'Transactions.Payment_id')
            ->leftJoin('Refunds', 'Payments.Payment_id', '=', 'Refunds.Payment_id');

        if ($request->User_id) {
            $query->where('Payments.User_id', $request->User_id);
        }
        if ($request->Payment_status) {
            $query->where('Payments.Payment_status', $request->Payment_status);
        }
        if ($request->start_date && $request->end_date) {
            $endDate = Carbon::parse($request->end_date)->endOfDay();
            $query->whereBetween('Payments.Created_at', [$request->start_date, $endDate]);
        } elseif ($request->start_date) {
            $query->where('Payments.Created_at', '>=', $request->start_date);
        }

        Log::debug('Raw SQL: ' . $query->toSql());
        Log::debug('Bindings: ' . json_encode($query->getBindings()));

        $userReports = $query->groupBy('Payments.User_id')->get();

        if ($userReports->isEmpty()) {
            Log::warning('No records found for report generation');
            return response()->json(['status' => 'error', 'message' => 'No records found'], 404);
        }

        $totalPayments = $userReports->sum('total_payments');
        $totalPaymentCount = $userReports->sum('payment_count');
        $paymentIds = Payment::whereIn('User_id', $userReports->pluck('User_id'))
            ->where(function ($q) use ($request) {
                if ($request->Payment_status) $q->where('Payment_status', $request->Payment_status);
                if ($request->start_date && $request->end_date) {
                    $endDate = Carbon::parse($request->end_date)->endOfDay();
                    $q->whereBetween('Created_at', [$request->start_date, $endDate]);
                } elseif ($request->start_date) $q->where('Created_at', '>=', $request->start_date);
            })
            ->pluck('Payment_id');

        $totalRefunded = Refund::whereIn('Payment_id', $paymentIds)->sum('Amount');
        $totalTransactionCount = Transaction::whereIn('Payment_id', $paymentIds)->count();

        try {
            $reportRecord = Report::create([
                'User_id' => Auth::id(), // Save under the authenticated user (e.g., 23)
                'Report_type' => 'payment_summary',
                'Total_amount' => $totalPayments,
                'Transaction_count' => $totalPaymentCount,
                'Payment_status' => $request->Payment_status ?? null,
                'Report_date' => now()->toDateTimeString(),
                'Created_at' => now(),
                'Updated_at' => now(),
            ]);
            Log::info('Report saved successfully: Report ID ' . $reportRecord->Report_id);
        } catch (\Exception $e) {
            Log::error('Failed to save report: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'Failed to save report: ' . $e->getMessage()], 500);
        }

        $userData = $userReports->map(function ($report) use ($paymentIds) {
            $userPaymentIds = Payment::where('User_id', $report->User_id)
                ->whereIn('Payment_id', $paymentIds)
                ->pluck('Payment_id');
            return [
                'User_id' => $report->User_id,
                'total_payments' => $report->total_payments,
                'payment_count' => $report->payment_count,
                'total_refunded' => Refund::whereIn('Payment_id', $userPaymentIds)->sum('Amount'),
                'transaction_count' => Transaction::whereIn('Payment_id', $userPaymentIds)->count(),
            ];
        })->all();

        $responseData = [
            'total_payments' => $totalPayments,
            'payment_count' => $totalPaymentCount,
            'total_refunded' => $totalRefunded,
            'transaction_count' => $totalTransactionCount,
            'report_id' => $reportRecord->Report_id,
            'user_data' => $userData,
        ];

        return response()->json(['status' => 'success', 'data' => $responseData], 200);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $report = Report::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:Users,User_id',
            'Report_type' => 'sometimes|in:payment_summary,user_activity',
            'Total_amount' => 'sometimes|numeric|min:0',
            'Transaction_count' => 'sometimes|integer|min:0',
            'Payment_status' => 'sometimes|in:pending,completed,failed',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $report->update($request->only(['User_id', 'Report_type', 'Total_amount', 'Transaction_count', 'Payment_status']));

        return response()->json(['status' => 'success', 'data' => $report], 200);
    }

    public function destroy($id): JsonResponse
    {
        $report = Report::findOrFail($id);
        $report->delete();

        return response()->json(['status' => 'success', 'message' => 'Report deleted successfully'], 200);
    }

    public function storeAnalytics(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Type' => 'required|in:user_engagement,job_placement,training_stats',
            'Data' => 'required|json',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $analytics = Analytics::create(['Type' => $request->Type, 'Data' => $request->Data]);
        return response()->json(['status' => 'success', 'data' => $analytics], 201);
    }

    public function getUserEngagement(Request $request): JsonResponse
    {
        $startDate = $request->input('start_date', now()->subMonth());
        $endDate = $request->input('end_date', now());

        $engagement = AuditLog::whereBetween('Created_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_actions'), DB::raw('COUNT(DISTINCT User_id) as active_users'))
            ->first();

        $videoProgress = DB::table('video_progress')
            ->whereBetween('updated_at', [$startDate, $endDate])
            ->select(DB::raw('AVG(progress) as avg_video_progress'), DB::raw('COUNT(*) as video_views'))
            ->first();

        $data = [
            'total_actions' => $engagement->total_actions ?? 0,
            'active_users' => $engagement->active_users ?? 0,
            'avg_video_progress' => $videoProgress->avg_video_progress ?? 0,
            'video_views' => $videoProgress->video_views ?? 0,
        ];

        Analytics::create(['Type' => 'user_engagement', 'Data' => json_encode($data)]);
        return response()->json(['status' => 'success', 'data' => $data], 200);
    }

    public function getJobPlacement(Request $request): JsonResponse
    {
        $startDate = $request->input('start_date', now()->subMonth());
        $endDate = $request->input('end_date', now());

        $placements = JobApplication::whereBetween('Applied_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_applications'), DB::raw('SUM(CASE WHEN Status = "hired" THEN 1 ELSE 0 END) as total_hires'))
            ->first();

        $interviews = Interview::whereBetween('Scheduled_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_interviews'))
            ->first();

        $data = [
            'total_applications' => $placements->total_applications ?? 0,
            'total_hires' => $placements->total_hires ?? 0,
            'total_interviews' => $interviews->total_interviews ?? 0,
        ];

        Analytics::create(['Type' => 'job_placement', 'Data' => json_encode($data)]);
        return response()->json(['status' => 'success', 'data' => $data], 200);
    }

    public function getTrainingStats(Request $request): JsonResponse
    {
        $startDate = $request->input('start_date', now()->subMonth());
        $endDate = $request->input('end_date', now());

        $courses = UserCourse::whereBetween('created_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_enrollments'), DB::raw('SUM(CASE WHEN Completion_status = "completed" THEN 1 ELSE 0 END) as total_completions'), DB::raw('AVG(Progress) as avg_progress'))
            ->first();

        $certifications = Certification::whereBetween('Issue_date', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_certifications'))
            ->first();

        $exams = DB::table('exam_attempts')
            ->whereBetween('Attempted_at', [$startDate, $endDate])
            ->select(DB::raw('AVG(Score) as avg_exam_score'), DB::raw('COUNT(*) as total_attempts'))
            ->first();

        $data = [
            'total_enrollments' => $courses->total_enrollments ?? 0,
            'total_completions' => $courses->total_completions ?? 0,
            'avg_progress' => $courses->avg_progress ?? 0,
            'total_certifications' => $certifications->total_certifications ?? 0,
            'avg_exam_score' => $exams->avg_exam_score ?? 0,
            'total_attempts' => $exams->total_attempts ?? 0,
        ];

        Analytics::create(['Type' => 'training_stats', 'Data' => json_encode($data)]);
        return response()->json(['status' => 'success', 'data' => $data], 200);
    }

    public function getHiringAnalytics(Request $request): JsonResponse
    {
        $companyId = Auth::user()->companyProfile->Company_id ?? null;
        if (!$companyId) return response()->json(['status' => 'error', 'message' => 'No company associated'], 403);

        $startDate = $request->input('start_date', now()->subMonth());
        $endDate = $request->input('end_date', now());

        $usage = CompanyFeatureUsage::where('Company_id', $companyId)
            ->whereBetween('Usage_date', [$startDate, $endDate])
            ->select(DB::raw('SUM(Job_postings_used) as total_postings_used'))
            ->first();

        $applications = JobApplication::whereHas('job', function ($q) use ($companyId) {
            $q->where('Company_id', $companyId);
        })->whereBetween('Applied_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_applications'), DB::raw('SUM(CASE WHEN Status = "hired" THEN 1 ELSE 0 END) as total_hires'))
            ->first();

        $interviews = Interview::where('Company_id', $companyId)
            ->whereBetween('Scheduled_at', [$startDate, $endDate])
            ->select(DB::raw('COUNT(*) as total_interviews'))
            ->first();

        $data = [
            'total_postings_used' => $usage->total_postings_used ?? 0,
            'total_applications' => $applications->total_applications ?? 0,
            'total_hires' => $applications->total_hires ?? 0,
            'total_interviews' => $interviews->total_interviews ?? 0,
        ];

        Report::create(['User_id' => Auth::id(), 'Type' => 'hiring_analytics', 'Data' => json_encode($data)]);
        return response()->json(['status' => 'success', 'data' => $data], 200);
    }

    public function getCandidateQuality(Request $request): JsonResponse
    {
        $companyId = Auth::user()->companyProfile->Company_id ?? null;
        if (!$companyId) return response()->json(['status' => 'error', 'message' => 'No company associated'], 403);

        $jobId = $request->input('job_id');
        $startDate = $request->input('start_date', now()->subMonth());
        $endDate = $request->input('end_date', now());

        $query = CandidateProfile::whereHas('user.jobApplications', function ($q) use ($companyId, $jobId) {
            $q->whereHas('job', function ($jq) use ($companyId) {
                $jq->where('Company_id', $companyId);
            });
            if ($jobId) $q->where('Job_id', $jobId);
        })->whereBetween('created_at', [$startDate, $endDate]);

        $quality = $query->select(DB::raw('AVG(Experience) as avg_experience'), DB::raw('COUNT(*) as total_candidates'))
            ->first();

        $skills = UserSkill::whereIn('User_id', $query->pluck('User_id'))
            ->select(DB::raw('GROUP_CONCAT(Skill) as common_skills'))
            ->groupBy('User_id')
            ->get();

        $certifications = Certification::whereIn('User_id', $query->pluck('User_id'))
            ->select(DB::raw('COUNT(*) as total_certifications'))
            ->first();

        $data = [
            'avg_experience' => $quality->avg_experience ?? 0,
            'total_candidates' => $quality->total_candidates ?? 0,
            'common_skills' => $skills->pluck('common_skills')->unique()->values(),
            'total_certifications' => $certifications->total_certifications ?? 0,
        ];

        Report::create(['User_id' => Auth::id(), 'Type' => 'candidate_quality', 'Data' => json_encode($data)]);
        return response()->json(['status' => 'success', 'data' => $data], 200);
    }

    public function exportReport(Request $request)
    {
        try {
            $reportId = $request->input('report_id');
            $type = $request->input('type');
            $startDate = $request->input('start_date', now()->subMonth());
            $endDate = $request->input('end_date', now());

            Log::info('Export Request: reportId=' . $reportId . ', type=' . $type . ', startDate=' . $startDate . ', endDate=' . $endDate);

            if ($type) {
                $reportData = $this->generateDynamicReport($type, $startDate, $endDate, $request);
                if (empty($reportData)) {
                    Log::warning('No data available for type: ' . $type);
                    return response()->json(['status' => 'error', 'message' => 'No data available for the specified type'], 404);
                }
                $report = new \stdClass();
                $report->data = $reportData;
            } elseif ($reportId) {
                $report = Report::find($reportId);
                if (!$report) {
                    Log::warning('Report not found for reportId: ' . $reportId);
                    return response()->json(['status' => 'error', 'message' => 'Report not found'], 404);
                }
            } else {
                Log::warning('No report_id or type provided');
                return response()->json(['status' => 'error', 'message' => 'Either report_id or type is required'], 400);
            }

            return Excel::download(new ReportsExport($report), 'report_' . ($reportId ?? $type) . '_' . now()->format('Ymd_His') . '.csv');
        } catch (\Exception $e) {
            Log::error('Export Report Error: ' . $e->getMessage());
            return response()->json(['status' => 'error', 'message' => 'An error occurred while exporting the report'], 500);
        }
    }

    private function generateDynamicReport($type, $startDate, $endDate, $request = null)
    {
        switch ($type) {
            case 'payment_summary':
                $query = Payment::query()
                    ->select('Payment_id', 'User_id', 'Amount', 'Payment_status', 'Created_at')
                    ->leftJoin('Transactions', 'Payments.Payment_id', '=', 'Transactions.Payment_id')
                    ->leftJoin('Refunds', 'Payments.Payment_id', '=', 'Refunds.Payment_id')
                    ->whereBetween('Payments.Created_at', [$startDate, $endDate]);
                if ($request && $request->input('Payment_status')) {
                    $query->where('Payments.Payment_status', $request->input('Payment_status'));
                }
                return $query->get()->map(function ($payment) {
                    return [
                        'Payment_id' => $payment->Payment_id,
                        'User_id' => $payment->User_id,
                        'Amount' => $payment->Amount,
                        'Payment_status' => $payment->Payment_status,
                        'Created_at' => $payment->Created_at,
                        'Refunded_Amount' => $payment->refunds->sum('Amount') ?? 0,
                        'Transaction_Count' => $payment->transactions->count() ?? 0,
                    ];
                })->all();
            default:
                return [];
        }
    }

    public function downloadReport(Request $request, $report_id)
    {
        Log::debug('Received Full Request: ' . json_encode($request->all()));
        Log::debug('Query String: ' . $request->getQueryString());
        Log::debug('All Input: ' . json_encode($request->input()));
        Log::debug('Path Report ID: ' . $report_id);

        if (!is_numeric($report_id)) {
            Log::warning('Invalid report_id: ' . json_encode($request->all()));
            return response()->json(['status' => 'error', 'message' => 'report_id must be a valid number'], 400);
        }

        $reportId = (int)$report_id;
        Log::info('Download Request: reportId=' . $reportId);

        $report = Report::find($reportId);
        if (!$report) {
            Log::warning('Report not found for reportId: ' . $reportId);
            return response()->json(['status' => 'error', 'message' => 'Report not found'], 404);
        }

        $reportDate = $report->Report_date ?: now()->subMonth();
        $paymentStatus = $report->Payment_status ?? 'completed';

        $paymentQuery = Payment::query()
            ->select(
                'Payments.Payment_id',
                'Payments.User_id',
                'Payments.Amount',
                'Payments.Payment_status',
                'Payments.Created_at'
            )
            ->leftJoin('Transactions', 'Payments.Payment_id', '=', 'Transactions.Payment_id')
            ->leftJoin('Refunds', 'Payments.Payment_id', '=', 'Refunds.Payment_id')
            ->where('Payments.User_id', $report->User_id)
            ->where('Payments.Created_at', '>=', $reportDate)
            ->where('Payments.Payment_status', $paymentStatus);

        Log::debug('Payment Query SQL: ' . $paymentQuery->toSql());
        Log::debug('Payment Query Bindings: ' . json_encode($paymentQuery->getBindings()));

        $paymentData = $paymentQuery->get()->map(function ($payment) {
            return [
                'Payment_id' => $payment->Payment_id,
                'User_id' => $payment->User_id,
                'Amount' => $payment->Amount,
                'Payment_status' => $payment->Payment_status,
                'Created_at' => $payment->Created_at,
                'Refunded_Amount' => optional($payment->refunds)->sum('Amount') ?? 0,
                'Transaction_Count' => optional($payment->transactions)->count() ?? 0,
            ];
        })->all();

        Log::debug('Payment Data: ' . json_encode($paymentData));

        if (empty($paymentData)) {
            Log::warning('No payment data found for reportId: ' . $reportId);
            return response()->json(['status' => 'error', 'message' => 'No data available for this report'], 404);
        }

        $reportData = new \stdClass();
        $reportData->data = $paymentData;

        $filename = 'report_' . $reportId . '_' . now()->format('Ymd_His') . '.csv';
        return Excel::download(new ReportsExport($reportData), $filename);
    }
}
